document.addEventListener('DOMContentLoaded', () => {
    const canvas = document.getElementById('gameCanvas');
    const ctx = canvas.getContext('2d');
    const startBtn = document.getElementById('startBtn');
    const restartBtn = document.getElementById('restartBtn');
    const gameOverScreen = document.getElementById('gameOverScreen');
    const finalScore = document.getElementById('finalScore');
    const scoreDisplay = document.getElementById('score');
  
    // Game variables
    let tileSize = 20;
    let tileCountX = canvas.width / tileSize;
    let tileCountY = canvas.height / tileSize;
    let snake = [];
    let food = {};
    let dx = tileSize;
    let dy = 0;
    let score = 0;
    let gamePaused = false;
    let gameOver = false;
    let interval;
  
    // Event listeners
    startBtn.addEventListener('click', startGame);
    restartBtn.addEventListener('click', restartGame);
    document.addEventListener('keydown', handleKeyPress);
  
    // Game loop
    function gameLoop() {
      if (!gamePaused && !gameOver) {
        clearCanvas();
        moveSnake();
        drawSnake();
        drawFood();
        if (checkCollision()) {
          endGame();
        }
        if (checkFoodCollision()) {
          eatFood();
        }
      }
    }
  
    // Initialize game
    function startGame() {
      startBtn.style.display = 'none';
      gameOverScreen.style.display = 'none';
      canvas.style.display = 'block';
      initSnake();
      placeFood();
      score = 0;
      scoreDisplay.textContent = 'Score: 0';
      gamePaused = false;
      gameOver = false;
      interval = setInterval(gameLoop, 100);
    }
  
    // Restart the game
    function restartGame() {
      gameOverScreen.style.display = 'none';
      clearInterval(interval);
      startGame();
    }
  
    // Initialize snake
    function initSnake() {
      snake = [
        { x: 3, y: 1 },
        { x: 2, y: 1 },
        { x: 1, y: 1 }
      ];
    }
  
    // Draw snake
    function drawSnake() {
      ctx.fillStyle = 'green';
      snake.forEach(segment => {
        ctx.fillRect(segment.x * tileSize, segment.y * tileSize, tileSize, tileSize);
      });
    }
  
    // Move snake
    function moveSnake() {
      const head = { x: snake[0].x + dx / tileSize, y: snake[0].y + dy / tileSize };
      snake.unshift(head);
      if (!checkFoodCollision()) {
        snake.pop();
      }
    }
  
    // Draw food
    function drawFood() {
      ctx.fillStyle = 'red';
      ctx.fillRect(food.x * tileSize, food.y * tileSize, tileSize, tileSize);
    }
  
    // Place food
    function placeFood() {
      food.x = Math.floor(Math.random() * tileCountX);
      food.y = Math.floor(Math.random() * tileCountY);
    }
  
    // Check collision with food
    function checkFoodCollision() {
      return snake[0].x === food.x && snake[0].y === food.y;
    }
  
    // Check collision with walls or itself
    function checkCollision() {
      return (
        snake[0].x < 0 ||
        snake[0].x >= tileCountX ||
        snake[0].y < 0 ||
        snake[0].y >= tileCountY ||
        snake.slice(1).some(segment => segment.x === snake[0].x && segment.y === snake[0].y)
      );
    }
  
    // Handle key presses
    function handleKeyPress(event) {
      if (event.key === 'ArrowUp' && dy === 0) {
        dy = -tileSize;
        dx = 0;
      } else if (event.key === 'ArrowDown' && dy === 0) {
        dy = tileSize;
        dx = 0;
      } else if (event.key === 'ArrowLeft' && dx === 0) {
        dx = -tileSize;
        dy = 0;
      } else if (event.key === 'ArrowRight' && dx === 0) {
        dx = tileSize;
        dy = 0;
      } else if (event.keyCode === 32) { // Space key
        gamePaused = !gamePaused;
      }
    }
  
    // Clearthe  canvas
    function clearCanvas() {
      ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
  
    // End the game
    function endGame() {
      clearInterval(interval);
      gameOver = true;
      gameOverScreen.style.display = 'block';
      finalScore.textContent = `Final Score: ${score}`;
    }
  
    // Eat the food
    function eatFood() {
      score++;
      scoreDisplay.textContent = `Score: ${score}`;
      placeFood();
    }
  });
  